const Joi = require('joi');

const enquirySchema = Joi.object({
    fullname: Joi.string()
        .pattern(/^[a-zA-Z\s'-]+$/)
        .min(3)
        .max(50)
        .required()
        .messages({
            'string.base': 'Full name must be a text.',
            'string.empty': 'Full name is required.',
            'string.min': 'Full name should have at least 3 characters.',
            'string.max': 'Full name should have at most 50 characters.',
            'string.pattern.base': 'Full name can only contain letters, spaces, apostrophes, and hyphens.',
        }),
    email: Joi.string()
        .email()
        .required()
        .messages({
            'string.base': 'Email must be a text.',
            'string.email': 'Please provide a valid email address.',
            'string.empty': 'Email is required.',
        }),
        contact: Joi.string()
        .min(10)
        .max(10)
        .pattern(/^[0-9]+$/)
        .required()
        .messages({
            'string.base': 'Contact number must be a number.',
            'string.empty': 'Contact number is required.',
            'string.min': 'Contact number should have at least 10 digits.',
            'string.max': 'Contact number should have at most 10 digits.',
            'string.pattern.base': 'Contact number must be a valid 10-digit number.',
        }),
    instagram_url: Joi.string()
        .uri()
        .required()
        .messages({
            'string.base': 'Instagram URL must be a text.',
            'string.empty': 'Instagram URL is required.',
            'string.uri': 'Please provide a valid Instagram URL.',
        }),
    date_of_birth: Joi.date()
        .required()
        .messages({
            'date.base': 'Date of birth must be a valid date.',
            'date.empty': 'Date of birth is required.',
        }),
    age: Joi.number()
        .min(1)
        .required()
        .messages({
            'number.base': 'Age must be a number.',
            'number.empty': 'Age is required.',
            'number.min': 'Age should be at least 1.',
        }),
    nationality: Joi.string()
        .min(2)
        .max(50)
        .required()
        .messages({
            'string.base': 'Nationality must be a text.',
            'string.empty': 'Nationality is required.',
            'string.min': 'Nationality should have at least 2 characters.',
            'string.max': 'Nationality should have at most 50 characters.',
        }),
    state: Joi.string()
        .min(2)
        .max(50)
        .required()
        .messages({
            'string.base': 'State must be a text.',
            'string.empty': 'State is required.',
            'string.min': 'State should have at least 2 characters.',
            'string.max': 'State should have at most 50 characters.',
        }),
        city: Joi.string()
        .min(2)
        .max(50)
        .required()
        .messages({
            'string.base': 'City must be a text.',
            'string.empty': 'City is required.',
            'string.min': 'City should have at least 2 characters.',
            'string.max': 'City should have at most 50 characters.',
        }),
    latitude: Joi.number()
        .required()
        .messages({
            'number.base': 'Latitude must be a number.',
            'number.empty': 'Latitude is required.',
        }),
    longitude: Joi.number()
        .required()
        .messages({
            'number.base': 'Longitude must be a number.',
            'number.empty': 'Longitude is required.',
        }),
    location_display: Joi.string()
        .required()
        .messages({
            'string.base': 'Location display must be a text.',
            'string.empty': 'Location display is required.',
        }),
    // portfolio is uploaded as PDF file (validated in validatePortfolioFile)

    created_at: Joi.date()
        .default(() => new Date())
        .messages({
            'date.base': 'Created date must be a valid date.',
        }),
});




function validateEnquiry(req, res, next) {
    // convert: true for multipart form string values; stripUnknown: true so "portfolio" (file field) in body is allowed and stripped
    const { error, value } = enquirySchema.validate(req.body, {
        convert: true,
        stripUnknown: true,
    });
    
    if (error) {
        return res.status(400).json({
            status: "error",
            message: error.details.map((err) => err.message),
        });
    }
    req.body = value;
    next();
}

function validatePortfolioFile(req, res, next) {
    if (!req.file) {
        return res.status(400).json({
            status: "error",
            message: ["Portfolio PDF file is required."],
        });
    }
    if (req.file.mimetype !== "application/pdf") {
        return res.status(400).json({
            status: "error",
            message: ["Portfolio must be a PDF file."],
        });
    }
    next();
}

module.exports = {
    validateEnquiry,
    validatePortfolioFile,
};








